import { NextResponse } from "next/server";
import { getDefaultLanguage, loadLanguages } from "./utils/serverUtils";

export async function middleware(request) {
  const token = request.cookies.get("token")?.value;
  const { pathname } = request.nextUrl;

  const personalInfo = request.cookies.get("personalInfo")?.value;
  const kycVerify = request.cookies.get("kycVerify")?.value;

  // Create ONE reusable response
  const response = NextResponse.next();

  // =========================
  // Locale cookie
  // =========================
  const locale = request.cookies.get("NEXT_LOCALE")?.value;

  if (!locale) {
    const languages = await loadLanguages();
    const defaultLang = getDefaultLanguage(languages)?.locale || "en";

    response.cookies.set("NEXT_LOCALE", defaultLang, {
      path: "/",
      maxAge: 60 * 60 * 24 * 365, // 1 year
    });
  }

  // =========================
  // Excluded paths
  // =========================
  if (
    pathname.startsWith("/_next") ||
    pathname.startsWith("/api/") ||
    pathname.startsWith("/static") ||
    pathname.startsWith("/lang") ||
    pathname.startsWith("/locales") ||
    pathname === "/favicon.ico" ||
    pathname === "/favicon.png" ||
    pathname.endsWith(".json") ||
    pathname.endsWith(".css") ||
    pathname.endsWith(".js") ||
    pathname.endsWith(".woff") ||
    pathname.endsWith(".woff2") ||
    pathname.endsWith(".eot")
  ) {
    return response;
  }

  // =========================
  // Allow public auth routes
  // =========================
  if (
    pathname.startsWith("/auth/register") ||
    pathname.startsWith("/auth/login/two-fa-verification-page") ||
    pathname.startsWith("/auth/step-verification")
  ) {
    return response;
  }

  // =========================
  // Logged in but visiting auth → redirect dashboard
  // =========================
  if (token && pathname.startsWith("/auth")) {
    const redirect = NextResponse.redirect(new URL("/dashboard", request.url));

    if (!locale) {
      const languages = await loadLanguages();
      const defaultLang = getDefaultLanguage(languages)?.locale || "en";

      redirect.cookies.set("NEXT_LOCALE", defaultLang, {
        path: "/",
        maxAge: 60 * 60 * 24 * 365,
      });
    }

    return redirect;
  }

  const protectedPaths = ["/dashboard", "/profile", "/settings"];

  // =========================
  // Not logged in → redirect to login
  // =========================
  if (!token && protectedPaths.some((p) => pathname.startsWith(p))) {
    const loginUrl = new URL("/auth/login", request.url);
    loginUrl.searchParams.set("callbackUrl", pathname);

    const redirect = NextResponse.redirect(loginUrl);

    if (!locale) {
      const languages = await loadLanguages();
      const defaultLang = getDefaultLanguage(languages)?.locale || "en";

      redirect.cookies.set("NEXT_LOCALE", defaultLang, {
        path: "/",
        maxAge: 60 * 60 * 24 * 365,
      });
    }

    return redirect;
  }

  // =========================
  // KYC / Personal Info Enforcement
  // =========================
  const needsVerification =
    protectedPaths.some((p) => pathname.startsWith(p)) &&
    (personalInfo !== "true" || kycVerify !== "true");

  if (token && needsVerification) {
    return NextResponse.redirect(
      new URL("/auth/step-verification/success-status", request.url)
    );
  }

  return response;
}

export const config = {
  matcher: [
    "/auth/login",
    "/dashboard/:path*",
    "/auth/:path*",
    "/dashboard",
    "/((?!api|_next/static|_next/image|favicon.ico|.*\\..*|locales).*)",
  ],
};
