"use client";

import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectLabel,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { useT } from "@/context/TranslationContext";
import ApiPath from "@/network/api/apiPath";
import NetworkService from "@/network/service/networkService";
import { Icon } from "@iconify/react";
import Link from "next/link";
import { useRouter } from "next/navigation";
import { useEffect, useState } from "react";
import Flatpickr from "react-flatpickr";
import { toast } from "react-toastify";

const CreateInvoice = () => {
  const network = new NetworkService();
  const [items, setItems] = useState([
    { id: 0, name: "", quantity: "", price: "", subtotal: "" },
  ]);
  const [to, setTo] = useState("");
  const [email, setEmail] = useState("");
  const [address, setAddress] = useState("");
  const [currency, setCurrency] = useState("");
  const [issueDate, setIssueDate] = useState("");
  const [published, setPublished] = useState("");
  const [wallets, setWallets] = useState([]);
  const [walletsLoading, setWalletsLoading] = useState(false);
  const [loading, setLoading] = useState(false);
  const router = useRouter();
  const t = useT();

  // Add new item with incremental ID
  const handleAddItem = () => {
    setItems((prev) => [
      ...prev,
      {
        id: prev.length > 0 ? prev[prev.length - 1].id + 1 : 0,
        name: "",
        quantity: "",
        price: "",
        subtotal: "",
      },
    ]);
  };

  // Remove item but keep at least one
  const handleRemoveItem = (id) => {
    if (items.length === 1) {
      toast.warning("At least one item must remain!");
      return;
    }
    setItems(items.filter((item) => item.id !== id));
  };

  // Update item fields
  const handleChange = (id, field, value) => {
    setItems((prev) =>
      prev.map((item) =>
        item.id === id
          ? {
              ...item,
              [field]: value,
              subtotal:
                field === "quantity" || field === "price"
                  ? (field === "quantity"
                      ? Number(value)
                      : Number(item.quantity)) *
                    (field === "price" ? Number(value) : Number(item.price))
                  : item.subtotal,
            }
          : item
      )
    );
  };

  // get wallets data
  const walletsData = async () => {
    try {
      setWalletsLoading(true);
      const res = await network.get(ApiPath.wallets);
      if (res.status === "completed") {
        setWallets(res.data.data.wallets);
      }
    } finally {
      setWalletsLoading(false);
    }
  };

  const handleCreateInvoice = async (e) => {
    e.preventDefault();
    if (
      !to ||
      !email ||
      !address ||
      !currency ||
      !issueDate ||
      published === ""
    ) {
      toast.error("Please fill in all required invoice fields.");
      return;
    }
    for (let i = 0; i < items.length; i++) {
      const item = items[i];
      if (!item.name || !item.quantity || !item.price) {
        toast.error(`Please fill in all fields for item ${i + 1}`);
        return;
      }
    }

    try {
      setLoading(true);
      const formData = new FormData();
      formData.append("to", to);
      formData.append("email", email);
      formData.append("address", address);
      formData.append("currency", currency);
      formData.append("issue_date", issueDate);
      formData.append("is_published", published);

      items.forEach((item, index) => {
        formData.append(`items[${index}][name]`, item.name);
        formData.append(`items[${index}][quantity]`, item.quantity);
        formData.append(`items[${index}][unit_price]`, item.price);
      });
      const res = await network.postFormData(ApiPath.createInvoice, formData);
      if (res.status === "completed") {
        toast.success("Invoice created successfully!");
        router.push("/dashboard/invoice");
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    walletsData();
  }, []);

  return (
    <div>
      <Link
        href="/dashboard/invoice"
        className="text-merchant-text text-[14px] font-bold flex items-center gap-2 hover:underline hover:text-merchant-primary transition-all duration-300 ease-in-out mb-3"
      >
        <Icon
          icon="lucide:arrow-left"
          width="20"
          height="20"
          className="w-[20px] h-[20px] rtl:rotate-180"
        />
        {t("buttonText.backButton")}
      </Link>
      <div className="max-w-[786px] mx-auto">
        <div className="mb-8">
          <h2 className="text-[20px] font-bold text-merchant-text mb-[30px]">
            {t("title.invoiceInformation")}
          </h2>
        </div>

        <form onSubmit={handleCreateInvoice}>
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-5 sm:gap-7 mb-8">
            <div>
              <div className="relative">
                <input
                  type="text"
                  id="to"
                  className={`user-input peer`}
                  placeholder=" "
                  value={to}
                  onChange={(e) => setTo(e.target.value)}
                />
                <label
                  htmlFor="to"
                  className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                >
                  {t("label.invoiceTo")}{" "}
                  <span className="text-merchant-error">*</span>
                </label>
              </div>
            </div>

            <div>
              <div className="relative">
                <input
                  type="email"
                  id="email"
                  className={`user-input peer`}
                  placeholder=" "
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                />
                <label
                  htmlFor="email"
                  className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                >
                  {t("label.emailAddress")}{" "}
                  <span className="text-merchant-error">*</span>
                </label>
              </div>
            </div>

            <div>
              <div className="relative">
                <input
                  type="text"
                  id="address"
                  className={`user-input peer`}
                  placeholder=" "
                  value={address}
                  onChange={(e) => setAddress(e.target.value)}
                />
                <label
                  htmlFor="address"
                  className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                >
                  {t("label.address")}{" "}
                  <span className="text-merchant-error">*</span>
                </label>
              </div>
            </div>

            <div className="relative">
              <Select
                value={currency}
                onValueChange={(value) => setCurrency(value)}
                disabled={walletsLoading}
              >
                <div className="absolute rtl:right-[16px] ltr:left-[16px] top-[-12px]">
                  <span className="text-[13px] text-[rgba(26,32,44,0.60)] font-semibold bg-white px-[10px]">
                    {t("placeholder.selectWallet")}{" "}
                    <span className="text-merchant-error">*</span>
                  </span>
                </div>
                <SelectTrigger className="w-full !h-[48px] border-2 border-merchant-input-border rounded-[8px] text-merchant-text focus:outline-none focus:shadow-outline">
                  <SelectValue
                    placeholder={
                      walletsLoading
                        ? t("placeholder.loadingWallets")
                        : t("placeholder.selectWallet")
                    }
                  />
                </SelectTrigger>

                <SelectContent className="rounded-[16px]">
                  <SelectGroup>
                    <SelectLabel>
                      {t("placeholder.availableWallets")}
                    </SelectLabel>
                    {wallets.map((wallet) => (
                      <SelectItem key={wallet.id} value={wallet.code}>
                        {wallet.name} ({wallet.formatted_balance} {wallet.code})
                      </SelectItem>
                    ))}
                  </SelectGroup>
                </SelectContent>
              </Select>
            </div>

            <div className="relative">
              <label className="user-label absolute rtl:right-[16px] ltr:left-[16px] top-[-10px] px-2">
                {t("label.issueDate")} <span className="text-red-500">*</span>
              </label>
              <Flatpickr
                className="border-2 border-merchant-input-border rounded-[8px] px-3 h-[48px] text-sm focus:outline-none focus:ring-1 focus:ring-transparent focus:border-merchant-primary w-full"
                placeholder={t("placeholder.selectDate")}
                options={{
                  dateFormat: "Y-m-d",
                  mode: "single",
                }}
                value={issueDate}
                onChange={([selectedDate]) => {
                  if (selectedDate) {
                    const formatted = selectedDate.toISOString().split("T")[0];
                    setIssueDate(formatted);
                  }
                }}
              />
            </div>

            <div className="relative">
              <Select
                value={published}
                onValueChange={(value) => setPublished(value)}
              >
                <div className="absolute rtl:right-[16px] ltr:left-[16px] top-[-12px]">
                  <span className="text-[13px] text-[rgba(26,32,44,0.60)] font-semibold bg-white px-[10px]">
                    {t("placeholder.selectStatus")}{" "}
                    <span className="text-merchant-error">*</span>
                  </span>
                </div>
                <SelectTrigger className="w-full !h-[48px] border-2 border-merchant-input-border rounded-[8px] text-merchant-text focus:outline-none focus:shadow-outline cursor-pointer">
                  <SelectValue placeholder={t("placeholder.selectStatus")} />
                </SelectTrigger>

                <SelectContent className="rounded-[16px]">
                  <SelectGroup>
                    <SelectLabel>{t("label.status")}</SelectLabel>
                    <SelectItem value="0">Draft</SelectItem>
                    <SelectItem value="1">Published</SelectItem>
                  </SelectGroup>
                </SelectContent>
              </Select>
            </div>
          </div>

          {/* Invoice Items Section */}
          <div className="mb-8">
            <div className="flex justify-between items-center mb-[30px]">
              <h2 className="text-[20px] font-bold text-merchant-text">
                {t("title.invoiceItems")}
              </h2>
              <button
                type="button"
                onClick={handleAddItem}
                className="h-[36px] flex items-center gap-[6px] border-2 border-merchant-primary rounded-[8px] px-[10px] text-[14px] font-bold text-merchant-text cursor-pointer transition-all duration-300 ease-in-out hover:bg-merchant-primary hover:text-white hover:border-merchant-primary"
              >
                <Icon icon="lucide:plus" width="20" height="20" />
                {t("buttonText.addItem")}
              </button>
            </div>

            {items.map((item) => (
              <div
                key={item.id}
                className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-5 sm:gap-6 mb-6 items-center"
              >
                <div>
                  <div className="relative">
                    <input
                      type="text"
                      id="item-name"
                      className={`user-input peer`}
                      placeholder=" "
                      value={item.name}
                      onChange={(e) =>
                        handleChange(item.id, "name", e.target.value)
                      }
                    />
                    <label
                      htmlFor="item-name"
                      className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                    >
                      {t("label.itemName")}{" "}
                      <span className="text-merchant-error">*</span>
                    </label>
                  </div>
                </div>
                <div>
                  <div className="relative">
                    <input
                      type="number"
                      id="quantity"
                      className={`user-input peer`}
                      placeholder=" "
                      value={item.quantity}
                      onChange={(e) =>
                        handleChange(item.id, "quantity", e.target.value)
                      }
                    />
                    <label
                      htmlFor="quantity"
                      className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                    >
                      {t("label.quantity")}{" "}
                      <span className="text-merchant-error">*</span>
                    </label>
                  </div>
                </div>
                <div>
                  <div className="relative">
                    <input
                      type="number"
                      id="unit-price"
                      className={`user-input peer`}
                      placeholder=" "
                      value={item.price}
                      onChange={(e) =>
                        handleChange(item.id, "price", e.target.value)
                      }
                    />
                    <label
                      htmlFor="unit-price"
                      className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                    >
                      {t("label.unitPrice")}{" "}
                      <span className="text-merchant-error">*</span>
                    </label>
                  </div>
                </div>
                <div>
                  <div className="relative">
                    <input
                      type="number"
                      id="subtotal"
                      className={`user-input peer`}
                      placeholder=" "
                      value={item.subtotal}
                      readOnly
                    />
                    <label
                      htmlFor="subtotal"
                      className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                    >
                      {t("label.subTotal")}{" "}
                      <span className="text-merchant-error">*</span>
                    </label>
                  </div>
                </div>
                <div className="">
                  <button
                    type="button"
                    onClick={() => handleRemoveItem(item.id)}
                    className="h-[48px] flex items-center justify-center gap-[8px] w-full bg-merchant-error hover:bg-red-700 text-white rounded-[8px] font-medium transition-all duration-200 ease-in-out cursor-pointer"
                  >
                    <Icon icon="lucide:trash-2" width="20" height="20" />
                    {t("buttonText.remove")}
                  </button>
                </div>
              </div>
            ))}
          </div>
          <div className="flex justify-start">
            <button
              className="group primary-button w-full"
              type="submit"
              disabled={loading}
            >
              <span className="primary-button-hover-effect"></span>
              <span className="primary-button-text">
                {loading
                  ? t("buttonText.creating")
                  : t("buttonText.createInvoice")}
              </span>
            </button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default CreateInvoice;
