import { RegisterInfoProvider } from "@/context/RegisterInfoContext";
import { SettingsProvider } from "@/context/settingsContext";
import { TranslationProvider } from "@/context/TranslationContext";
import { UserProvider } from "@/context/UserContext";
import { getDictionary } from "@/lib/i18n/get-dictionary";
import { getLocale } from "@/lib/i18n/get-locale";
import { getSelectedLanguage, loadLanguages } from "@/utils/serverUtils";
import { getSetting } from "@/utils/utils";
import { Plus_Jakarta_Sans } from "next/font/google";
import { ToastContainer } from "react-toastify";
import "./globals.css";

//font
const plusJakartaSans = Plus_Jakarta_Sans({
  subsets: ["latin"],
  variable: "--font-plus-jakarta-sans",
});

export async function generateMetadata() {
  const apiBaseUrl = process.env.NEXT_PUBLIC_API_BASE_URL;

  if (!apiBaseUrl) {
    return {
      title: "Merchant",
      description: "Empower Your Financial Journey",
      icons: { icon: "/favicon.png" },
    };
  }

  try {
    const res = await fetch(`${apiBaseUrl}/get-settings`, {
      next: { revalidate: 3600 },
    });
    const data = await res.json();
    const settings = data?.data || [];
    const siteFavicon = getSetting(settings, "site_favicon");
    const siteTitle = getSetting(settings, "site_title");

    return {
      title: siteTitle || "Merchant",
      description: "Empower Your Financial Journey",
      icons: {
        icon: siteFavicon || "/favicon.png",
      },
    };
  } catch (err) {
    console.log(err);
    return {
      title: "Merchant",
      description: "Empower Your Financial Journey",
      icons: { icon: "/favicon.png" },
    };
  }
}

export default async function RootLayout({ children }) {
  const locale = await getLocale();
  const dictionary = await getDictionary(locale);
  const languages = await loadLanguages();
  const selectedLanguage = getSelectedLanguage(languages, locale);
  return (
    <html
      lang={locale}
      className={`${plusJakartaSans.variable}`}
      dir={selectedLanguage.is_rtl ? "rtl" : "ltr"}
    >
      <body className={plusJakartaSans.className}>
        <ToastContainer
          position="top-right"
          autoClose={3000}
          hideProgressBar={false}
          newestOnTop={false}
          closeOnClick
          rtl={false}
          pauseOnFocusLoss
          draggable
          pauseOnHover
        />
        <TranslationProvider
          initialLang={locale}
          initialTranslations={dictionary}
        >
          <SettingsProvider>
            <UserProvider>
              <RegisterInfoProvider>{children}</RegisterInfoProvider>
            </UserProvider>
          </SettingsProvider>
        </TranslationProvider>
      </body>
    </html>
  );
}
